'-------------------------------------------------------------------------
' Wiznet W5100 Micromite Examples
'
' Hardware Connections
'
' +--------+--------------------------------------------+
' | Wiznet | Micromite                                  |
' +--------+--------------------------------------------+
' | Ground |  Pin 19                                    |
' | +5V    |  +5V                                       |
' | RST    |  Pin 23 (Wiznet W5100 Reset)               |
' | SS     |  Pin 15 (SPI Slave Select)                 |
' | CLK    |  Pin 25 (SPI Clock)                        |
' | MO     |  Pin 3 (SPI MOSI - Master Out, Slave In)   |
' | MI     |  Pin 14 (SPI MISO - Master In, Slave Out)  |
' | Gnd    |  not used                                  |
' | PoE+   |  not used                                  |
' | PoE-   |  not used                                  |
' +--------+--------------------------------------------+

Option explicit
Option base 1

dim ip1%, ip2%, ip3%, ip4%, dport%, looptimer%, i%, char%

' Call with pin used for Wiznet Reset and SPI Slave Select of Wiznet:
WN.Init 23, 15

' Set a pin as analog input for testing webserver
SetPin 26, ain

' Set up the IP address to use
ip1%=192
ip2%=168
ip3%=1
ip4%=65

' Set static MAC address/IP address/subnet mask/gateway

WN.SetMAC &H00,&HFE,ip1%,ip2%,ip3%,ip4%
WN.SetIP ip1%,ip2%,ip3%,ip4%
WN.SetMask 255,255,255,0
WN.SetGateway ip1%,ip2%,ip3%,1

WN.Start    ' Sets various internal registers for Wiznet

If WN.SockUDPInit(3, 64000) Then
  Print "Socket initialized on local port 64000"
Else
  Error "Failed to initialize socket"
End If

If WN.SockUDPInit(2, 124) Then
  Print "Socket initialized on local port 124"
Else
  Error "Failed to initialize socket"
End If

WN.SockUDPDest 3, 192, 168, 1, 14, 124

do
  WN.SockWrite 3,"Test UDP!!! "+str$(TIMER)+WN_CrLf$
  looptimer%=timer
  do
    For i%=1 To WN.sockData(2)
      char%=WN.sockRead(2)
      Print Chr$(char%);
    next i%
    if timer-looptimer%>5000 then exit DO
  loop
loop

End

'-------------------------------------------------------------------------
' WIZNET W5100 Subs and Functions

Sub WN.Init(resetPin%, slaveSelectPin%)
  dim integer WN_OptionBase=1       ' Set to match option base above (if changed)
  dim WN_CrLf$=Chr$(&H0D)+Chr$(&H0A)
  dim WN_SPISpeed%=1000000 '10000000 is max that works
  DIM WN_ResetPin%=resetPin%
  DIM WN_SSPin%=slaveSelectPin%

  ' Set the slave select pin HIGH and to output pin
  Pin(WN_SSPin%)=1
  SetPin WN_SSPin%, dout

  SetPin WN_ResetPin%, dout
  Pin(resetPin%)=0
  ' Wiznet docs: assert low for 2us to reset Wiznet
  Pause 10 ' we wait 10 milliseconds
  Pin(resetPin%)=1
End Sub

Sub WN.Close()
  PIN(WN_SSPin%)=0
  SETPIN WN_SSPin%,OFF
  PIN(WN_ResetPin%)=0  
  SETPIN WN_ResetPin%,OFF
END SUB
  
Sub WN.Start()
  WN.Write(0, 0, &B00000000) ' Mode Register, enable ping
  WN.Write(0, &H16, &B00000000) ' Interrupt Mask Register
  WN.Write(0, &H17, &H0F) ' Retry time value high (400ms)
  WN.Write(0, &H18, &HA0) ' Retry time value low (400ms)
  WN.Write(0, &H19, &H08) ' retry count register
  WN.Write(0, &H1A, &B01010101) ' RX Memory Size (2KB/socket)
  WN.Write(0, &H1B, &B01010101) ' TX Memory Size (2KB/socket)
End Sub
  
Function WN.Read(addrHi%, addrLo%)
  Local junk%
  Pin(WN_SSPin%)=0
  SPI open WN_SPISpeed%, 0, 8
  junk%=SPI(&H0F)
  junk%=SPI(addrHi%)
  junk%=SPI(addrLo%)
  WN.Read=SPI(0)
  Pin(WN_SSPin%)=1
  SPI close
End Function
  
Sub WN.Write(addrHi%, addrLo%, writeData%)
  Local junk%
  Pin(WN_SSPin%)=0
  SPI open WN_SPISpeed%, 0, 8
  junk%=SPI(&HF0)
  junk%=SPI(addrHi%)
  junk%=SPI(addrLo%)
  junk%=SPI(writeData%)
  Pin(WN_SSPin%)=1
  SPI close
End Sub
  
Sub WN.SetGateway (oct1%, oct2%, oct3%, oct4%)
  WN.Write(&H00, &H01, oct1%)
  WN.Write(&H00, &H02, oct2%)
  WN.Write(&H00, &H03, oct3%)
  WN.Write(&H00, &H04, oct4%)
End Sub
  
Sub WN.SetMask (oct1%, oct2%, oct3%, oct4%)
  WN.Write(&H00, &H05, oct1%)
  WN.Write(&H00, &H06, oct2%)
  WN.Write(&H00, &H07, oct3%)
  WN.Write(&H00, &H08, oct4%)
End Sub
  
Sub WN.SetMAC (oct1%, oct2%, oct3%, oct4%, oct5%, oct6%)
  WN.Write(&H00, &H09, oct1%)
  WN.Write(&H00, &H0A, oct2%)
  WN.Write(&H00, &H0B, oct3%)
  WN.Write(&H00, &H0C, oct4%)
  WN.Write(&H00, &H0D, oct5%)
  WN.Write(&H00, &H0E, oct6%)
End Sub
  
Sub WN.SetIP (oct1%, oct2%, oct3%, oct4%)
  WN.Write(&H00, &H0F, oct1%)
  WN.Write(&H00, &H10, oct2%)
  WN.Write(&H00, &H11, oct3%)
  WN.Write(&H00, &H12, oct4%)
End Sub
  
Function WN.GetIP$()
  Local oct1%, oct2%, oct3%, oct4%
  oct1%=WN.Read(&H00, &H0F)
  oct2%=WN.Read(&H00, &H10)
  oct3%=WN.Read(&H00, &H11)
  oct4%=WN.Read(&H00, &H12)
  WN.GetIP$=Str$(oct1%)+"."+Str$(oct2%)+"."+Str$(oct3%)+"."+Str$(oct4%)
End Function
  
Function WN.SockTCPInit(socket%, srcport%)
  WN.Write(socket%+4, &H00, &H01) 'Sn_MR Mode Register
  WN.Write(socket%+4, &H04, srcport% >> 8) 'Sn_PORT Source Port Hi
  WN.Write(socket%+4, &H05, srcport% And &HFF) 'Sn_PORT Source Port Lo
  WN.Write(socket%+4, &H01, &H01) 'Sn_CR Command Register, Open
  'Sn_SR Status Register
  If WN.Read(socket%+4, &H03)<>&H13 Then
    'Failed to init socket
    WN.SockTCPInit=0
    WN.Write(socket%+4, &H01, &H10) 'Sn_CR Command Register, Close
  Else
    WN.SockTCPInit=1
  End If
End Function
  
Function WN.SockUDPInit(socket%, srcport%)
  WN.Write(socket%+4, &H00, &H02) 'Sn_MR Mode Register
  WN.Write(socket%+4, &H04, srcport% >> 8) 'Sn_PORT Source Port Hi
  WN.Write(socket%+4, &H05, srcport% And &HFF) 'Sn_PORT Source Port Lo
  WN.Write(socket%+4, &H01, &H01) 'Sn_CR Command Register, Open
  'Sn_SR Status Register
  If WN.Read(socket%+4, &H03)<>&H22 Then
    'Failed to init socket
    WN.SockUDPInit=0
    WN.Write(socket%+4, &H01, &H10) 'Sn_CR Command Register, Close
  Else
    WN.SockUDPInit=1
  End If
End Function
  
Function WN.SockTCPListen(socket%)
  WN.Write(socket%+4, &H01, &H02) 'Sn_CR Command Register, Listen
  'Sn_SR Status Register
  If WN.Read(socket%+4, &H03)<>&H14 Then
    'Failed to listen
    WN.SockTCPListen=0
    WN.Write(socket%+4, &H01, &H10) 'Sn_CR Command Register, Close
  Else
    WN.SockTCPListen=1
  End If
End Function
  
Sub WN.SockClose(socket%)
  WN.Write(socket%+4, &H01, &H08) 'Sn_CR Command Register, DISCON
End Sub
  
Function WN.SockEstablished(socket%)
  Local SnCR%
  SnCR%=WN.Read(socket%+4, &H03)
  If SnCR%=&H1C Then
    WN.Write(socket%+4, &H01, &H08) 'Sn_CR Command Register, DISCON
  End If
  If SnCR%=&H17 Then
    WN.SockEstablished=1
  Else
    WN.SockEstablished=0
  End If
End Function
  
Function WN.SockClosed(socket%)
  Local SnCR%
  SnCR%=WN.Read(socket%+4, &H03)
  If SnCR%=&H00 Then
    WN.SockClosed=1
  Else
    WN.SockClosed=0
  End If
End Function
  
Function WN.SockData(socket%)
  WN.SockData=(WN.Read(socket%+4, &H26)*256)+(WN.Read(socket%+4, &H27))
End Function
  
Function WN.SockRead(socket%)
  Local SnRXRead%, offset%, address%
  SnRXRead%=(WN.Read(socket%+4, &H28)*256)+(WN.Read(socket%+4, &H29))
  offset%=SnRXRead% And &H07FF
  address%=&H6000+(socket%*&H800)+offset%
  WN.SockRead=WN.Read(address% >> 8, address% And &HFF) 'readbyte
  SnRXRead%=SnRXRead%+1
  If SnRXRead%>65535 Then SnRXRead%=0
  WN.Write(socket%+4, &H28, SnRXRead% >> 8) 'increment counter
  WN.Write(socket%+4, &H29, SnRXRead% And &HFF)
  WN.Write(socket%+4, &H01, &H40) 'Sn_CR Command Register, RECV
End Function
  
SUB WN.SockUDPDest (socket%, doct1%, doct2%, doct3%, doct4%, dstport%)
  WN.Write(socket%+4, &H0C, doct1%)
  WN.Write(socket%+4, &H0D, doct2%)
  WN.Write(socket%+4, &H0E, doct3%)
  WN.Write(socket%+4, &H0F, doct4%)
  WN.Write(socket%+4, &H10, dstport% >> 8) 'Sn_PORT Source Port Hi
  WN.Write(socket%+4, &H11, dstport% And &HFF) 'Sn_PORT Source Port Lo
END SUB
  
Sub WN.SockWrite(socket%, sockTX$)
  Local TXchars%, SnTXFSR%, SnTXWR%, offset%, address%, i%
  TXchars%=Len(sockTX$)
waitforprocessing:
  SnTXFSR%=(WN.Read(socket%+4, &H20)*256)+(WN.Read(socket%+4, &H21))
  If SnTXFSR%<TXchars% Then GoTo waitforprocessing
  SnTXWR%=(WN.Read(socket%+4, &H24)*256)+(WN.Read(socket%+4, &H25))
  offset%=SnTXWR% And &H07FF
  address%=&H4000+(socket%*&H800)+offset%
  For i%=1 To TXchars%
    If address%>&H4000+(socket%*&H800)+&H7FF Then address%=address%-&H800
    WN.Write(address%>>8, address% And &HFF, Asc(Mid$(sockTX$, i%, 1)))
    address%=address%+1
  Next i%
  SnTXWR%=SnTXWR%+TXchars%
  If SnTXWR%>65535 Then SnTXWR%=SnTXWR%-65536
  WN.Write(socket%+4, &H24, SnTXWR% >> 8)
  WN.Write(socket%+4, &H25, SnTXWR% And &HFF)
  WN.Write(socket%+4, &H01, &H20) 'Sn_CR Command Register, SEND
End Sub
  
Function WN.SockTCPConnect(socket%, oct1%, oct2%, oct3%, oct4%, destport%)
  Local timerstart%, SnCR%
  WN.Write(socket%+4, &H0C, oct1%) 'Destination IP Address
  WN.Write(socket%+4, &H0D, oct2%)
  WN.Write(socket%+4, &H0E, oct3%)
  WN.Write(socket%+4, &H0F, oct4%)
  WN.Write(socket%+4, &H10, (destport% >> 8) 'Sn_DPORT Hi
  WN.Write(socket%+4, &H11, destport% And &HFF) ' Lo
  WN.Write(socket%+4, &H01, &H04) 'Sn_CR Command Register, Connect
  WN.SockTCPConnect=0
  timerstart% = Timer
  Do While (Timer - timerstart%)<4000 ' Wait at least 4 seconds for connection
    SnCR%=WN.Read(socket%+4, &H03)
    If SnCR%=&H17 Then ' Established
      WN.SockTCPConnect=1
      Exit Do
    Else If SnCR%=&H00 Then ' Closed (Timeout connecting)
      Exit Do
    End If
  Loop
End Function
